/////////////////////////////////////////////////////////////////////////////////
// License: Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
/////////////////////////////////////////////////////////////////////////////////
//
// This shader is derived from TheEmu's  "Swarms of Butterflies"  shaders  which
// can be found in TheEmu's "Experiments with Swarms" series of scenes.  It  has
// been modified for EverthangForever such that the sizes of the butterflies now
// change as they circulate such that those closer to the camera are larger than
// those further away and their collective motions have been made more chaotic.
//
// These shaders are all based on e#21916.3 obtained from  GlslSandbox.com,  for
// which the original author is not known, but have been rewritten by TheEmu for
// use with VGHD and at the same time have been both slightly tidied up and made
// more easily configurable.
//
/////////////////////////////////////////////////////////////////////////////////

// Shader inputs.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// A scene comprises several layers at different depths.  This shader is normally
// used to provide a swarm circling a performer by using three of  these  layers,
// one for the swarm elements in front of the performer,  one  for  the performer
// and one for the swarm elements behind the performer.  The  depth  parameter is
// used to distinguish these layers.  Currently  a depth of 1 or more corresponds
// to being in front of the performer and 0 or less to being behind and the  user
// should only use 0 or 1. A future development may introduce more levels.

uniform int depth; // The relative depth of the layer in the scene

// By default the swarm elements do not change their size  on  the  screen  while
// they circulate.  This behaviour can be modified so that the apparent sizes are
// largest when closest to the camera and smallest when further away by  using  a
// non zero value for the following parameter, cosThetaZFactor, which should take
// values between 0.0 and 1.0.

uniform float cosThetaZFactor; // Z scaling dependancy on cos(theta)

// In addition to the scaling due to the variation in distance that should result
// from the circling motion a second "Z scaling" factoe has been introduced. This
// is actually just an overall scale factor that is a function of time but it the
// effect of causing the the overall swarm and swarm element sizes all to vary by
// the same proportion - which in turn makes it appear as if the swarm itself is
// moving into and out of the tunnel.  This only works if the swarm is cetered on
// the center of the tunnel, if it is significantly off center then the  illusion
// of moving into and out of the tunnel is lost and instead it looks  as  if  the
// butterflies themselves change size. The default value, 0., for zScaleFrequency
// means that there is no variation in the swarm due to this  factor.  Reasonable
// values are from 0.0 to 0.5 as higher values cause the butterflies to move back
// and forth along the tunnel too rapidly. 

uniform float zScaleFrequency; // Z scaling frequency

/////////////////////////////////////////////////////////////////////////////////

// The number of swarm size and its position are controlled by  the  following
// set of parameters. Normaly these will all be constants,  but  they may also
// be expressions involving shader variables visible where the parameters  are
// used and hence may depend on time or position.

float SWARM_SCALE = abs(cos(u_Elapsed*zScaleFrequency));

#define SWARM_SIZE 8 // Number of elements in the swarm.

//#define SWARM_CENTER vec2(0.0,0.0) // Position of the swarm's center.
#define SWARM_CENTER vec2(0.0,0.75) // Position of the swarm's center.
#define SWARM_RADIUS vec2(0.5,0.5) // "Physical" size of the swarm.

// The colour of the swarm elements and the way in which their glow fades with
// with distance from their individual centers are controled by the  following
// set of parameters.  The arguments i and r are the element index, which runs
// from 0 to SWARM_SIZE-1,  and  the position of the current pixel relative to
// the center of the element.

// In this particular example the swarm elements are butterflies.  The  method
// used for the butterflies uses a formula for a curve describing the  outline
// of a butterfly shaped area based on polar co-ordinates and simply  checking
// if a point lies within this curve or not.
//swapped colors around from original for front butterflies

//const vec3 butterfly_a_colour_1 = vec3(1.0,1.0,1.0);
const vec3 butterfly_a_colour_1 = vec3(0.90,0.10,0.10);
const vec3 butterfly_a_colour_2 = vec3(0.1,0.1,1.0);

//const vec3 butterfly_b_colour_1 = vec3(0.90,0.10,0.10);
const vec3 butterfly_b_colour_1 = vec3(1.0,1.0,1.0);
const vec3 butterfly_b_colour_2 = vec3(0.15,0.15,0.77);

//const vec3 butterfly_c_colour = vec3(1.0,1.0,0.0);
const vec3 butterfly_c_colour = vec3(0.6,0.6,1.0);
//const vec3 butterfly_d_colour = vec3(0.6,0.6,1.0);
const vec3 butterfly_d_colour = vec3(1.0,1.0,0.0);

vec3 butterfly_a(int i,vec2 r,float theta,float zeta)
 { float cosTheta = cos(theta);
   // Convert r to polar coordinates (p,d)
   float p = atan(r.y,r.x) + zeta*cosTheta;
   float d = length(r);
   // Generate the outline of a butterfly.
   float a = 7.0 - 0.5*sin(p) + 2.5*sin(3.0*p) + 2.0*sin(5.0*p) - 1.7*sin(7.0*p)
                              + 3.0*cos(2.0*p) - 2.0*cos(4.0*p) - 0.4*cos(16.0*p);
   a = a * ( 1.0 + cosTheta*cosThetaZFactor ) * SWARM_SCALE / 400.0;
   // Assign colour
   if ( d > a ) return butterfly_a_colour_1/300.0/d;
   if ( (d-a/2.0) < 0.4*a ) return butterfly_a_colour_2;
  // return butterfly_a_colour_1; // this floods the quad with color. Instead use..
return vec3(0.0);
 }

vec3 butterfly_b(int i,vec2 r,float theta,float zeta)
 { float cosTheta = cos(theta);
   // Convert r to polar coordinates (p,d)
   float p = atan(r.y,r.x) + zeta*cosTheta;
   float d = length(r);
   // Generate the outline of a butterfly.
   float a = 8.0 - sin(p) + 2.0*sin(3.0*p) + 2.0*sin(5.0*p) - sin(7.0*p)
                          + 3.0*cos(2.0*p) - 2.0*cos(4.0*p);
   a = a * ( 1.0 + cosTheta*cosThetaZFactor ) * SWARM_SCALE / 400.0;
   // Assign colour
   if ( d > a ) return butterfly_b_colour_1/300.0/d;
   if ( (d-a/2.0) < 0.42*a ) return butterfly_b_colour_2;
  // return butterfly_b_colour_1; // this floods the quad with color. Instead use..
return vec3(0.0);
 }

vec3 butterfly_c(int i,vec2 r,float theta,float zeta)
 { float cosTheta = cos(theta);
   // Convert r to polar coordinates (p,d)
   float p = atan(r.y,r.x) + zeta*cosTheta;
   float d = length(r);
   // Generate the outline of a butterfly.
   float a = 8.0 - sin(p) + 2.0*sin(3.0*p) + 2.0*sin(5.0*p) - sin(7.0*p)
                          + 3.0*cos(2.0*p) - 2.0*cos(4.0*p);
   a = a * ( 1.0 + cosTheta*cosThetaZFactor ) * SWARM_SCALE / 400.0;
   // Assign colour
   return d < a ? butterfly_c_colour : butterfly_c_colour/200.0/d;
 }

vec3 butterfly_d(int i,vec2 r,float theta,float zeta)
 { float cosTheta = cos(theta);
   // Convert r to polar coordinates (p,d)
   float p = atan(r.y,r.x) + zeta*cosTheta;
   float d = length(r);
   // Generate the outline of a butterfly.
   float a = 7.0 - 0.5*sin(p) + 2.5*sin(3.0*p) + 2.0*sin(5.0*p) - 1.7*sin(7.0*p)
                              + 3.0*cos(2.0*p) - 2.0*cos(4.0*p) - 0.4*cos(16.0*p);
   a = a * ( 1.0 + cosTheta*cosThetaZFactor ) * SWARM_SCALE / 400.0;
   // Assign colour
   return d < a ? butterfly_d_colour : butterfly_d_colour/200.0/d;
 }
vec3 butterfly(int i, vec2 r, float theta, float y)
 { float zeta = float(mod(i,2))*2.0 - 1.0;
   switch (int(mod(i,4)))
    { case 0 : return butterfly_a(i,r,theta,zeta);
      case 1 : return butterfly_b(i,r,theta,zeta);
      case 2 : return butterfly_c(i,r,theta,zeta);
      case 3 : return butterfly_d(i,r,theta,zeta);
    }
 }

#define SWARM_ELEMENT(i,r) butterfly(i,r,theta,y);

// The way in which the swarm circulates are controlled by the  following  set
// of parameters.  See the main body of the shader for their meanings.  It  is
// possible to specify a different set of these parameters for each element of
// the swarm, i identifies the element and runs from 0 to SWARM_SIZE-1.

// A slightly complex form is used for ALPHA such that ALPHA has the following
// properties:-
//
//   It is positive for odd values of i,  and negative for even values.
//   This causes half the butterflies to circle in one direction  while
//   the other half circle in the opposite direction.
//
//   The speeds at which the butterfies circle all differ but only by a
//   relatively small amount.  This causes their relative motions to be
//   somewhat chaotic and so the butterflies do not periodicaly arrange
//   themselves into lines as was the case with the original shader.

//#define ALPHA_1(i) ( (float(mod(i,2)) - 0.5) * 2.0 ) // The time scale factor
#define ALPHA_1(i) ( (float(mod(i,2)) - 0.5) * 1.42 ) // The time scale factor
#define ALPHA_2(i) ( float(i) / ( 50.0+ALPHA_1(i)) ) // for the ALPHA angle
#define ALPHA(i)   ( 0.8*(ALPHA_1(i)+ALPHA_2(i)) )   // as a function of i.

#define BETA(i)  ( ALPHA(i)*0.01 ) // Time scale factor for the angle beta.
#define GAMMA(i) ( 77.0 - 11.0*i ) // Scale factor for the factor gamma.
#define THETA(i) ( 0.00 )          // Initial value for the theta angle.

// Normally this shader is used twice, once to generate that part of the swarm
// that is in behind the VGHD clip and a second time to generate the part that
// is in front of it. The simplest way to select which elements to generate is
// to make he decision based on their theta angle which this is positive for a
// half of the orbit and negative for the other half. As a special case values
// of depth greater than 999 cause the swarm element to be generated no matter
// what value the theta angle has - this permits a single shader to be used in
// place of a pair where the butterflies are not circling round something  but 
// are all always in front or all always behind other scene elements.  However
// if you do this then there will be times when some of the  butterflies  that
// in the back part of their orbit will appear to be in front of  others  that
// are in the front part of the orbit. Whether this matters depends on the way
// the shader is used in the scene,  in particular you probably should not use
// the ability to change the size of the butterflies as they circle. 

#define GENERATE_ELEMENT(i,theta) ( (depth>999) || ( (depth>0)^^(cos(theta)<0.0) ) )

// The following parameters modify the x and y coordinates. There is no reason
// to define them as non zero constants as SWARM_CENTER already handles  that.
// However,  they may be defined as expressions involving time, coordinates or
// internal variables of the shader such as theta to cause the swarm  elements
// to bob up and down or side to side as they orbit.

// This particular pair of DELTA parameters cause the swarm elements to bob up
// and down sinusoidaly with those in the middle bobbing most strongly and for
// the orbits to be wide in the middle and smaller at the top and bottom.

//#define DELTA_X pow(abs(y)*1.1-0.1,4.0)*sin(theta)/50.0
#define DELTA_X pow(abs(y)*1.1-0.1,4.0)*sin(theta)/70.0
//#define DELTA_Y sin(theta*(0.503-abs(y-0.5))*11.0)/10.0
#define DELTA_Y sin(theta*(0.503-abs(y-0.5))*11.0)/60.0

/////////////////////////////////////////////////////////////////////////////////
//
// This shader generates a swarm of glowing spheres which may be considered as
// rotating around around a cylinder with the center of each at a fixed height
// above the cylinder's base and at an angle theta around its circumference.
//
// Theta is given by
//
//    theta = THETA + ALPHA*t + sin(BETA*t)*GAMMA
//
// where t is the time and the other parameters are, nominaly,  constants.  As
// a result the whole swarm circles the cylinder at a speed  given  by  ALPHA,
// but the relative phases of each element varies depending on the  values  of
// BETA and GAMMA.  The result is that at times the swarm elements appear as a
// single starnd of lights,  at others as two or more strands and at yet other
// times there are no discernable strands.
//
// The shader is controlled by a number of parameters,  including ALPHA,  BETA
// and GAMMA, which are described above.  Although nominaly constant the value
// of any parameter can be defined as a function of time or position  allowing
// a great deal of flexibility. For example the spheres may be made to pulsate
// or their colours may be made to vary as time progresses.
//
/////////////////////////////////////////////////////////////////////////////////

void main( void )
 {

   // Get the extent of the swarm in the X and Y directions.

   vec2 swarm_scale = SWARM_RADIUS * SWARM_SCALE;

   // Get coordinates of the current pixel such that X increases from  X  from
   // left to right from -1.0 at the left edge to +1.0 at the right  edge  and
   // Y increases from 0.0 at the bottom to 1.0 at the top of the screen.

   vec2 pixel = gl_FragCoord.xy/u_WindowSize.xy;
   pixel = ( pixel - 0.5 ) * 2.0;

   // Adjust the coordinates to be centered with the swarm and to go from -1.0
   // to +1.0 at the edges of the swarm. Working in this coordinate system  is
   // slightly simpler than working in the screen coordinates themselves.

   pixel = pixel / swarm_scale + SWARM_CENTER;

   // Initialise the colour for this pixel.

   vec3 colour = vec3(0.0);

   // Process each swarm element in turn.

   for ( int i = 0; i < SWARM_SIZE; i++ )
    {
      // Calcualate the alpha, beta and gamma angles.

      float alpha = ALPHA(i) * u_Elapsed;
      float beta  = BETA(i)  * u_Elapsed;
      float gamma = GAMMA(i) * sin(beta);

      // Get swarm element i's y coordinate and theta angle.

      float y = float(i-SWARM_SIZE/2) * 2.0 / float(SWARM_SIZE-1);
      float theta = THETA(i) + alpha + gamma*y;

      // If this swarm element is to be generated then do so.

      if ( GENERATE_ELEMENT(i,theta) )
       {
         // Get the distance between the element and the  pixel.
         // Multiplying by swarm_scale compensates for the swarm
         // and the screen coordinate systems  having  different
         // scales for each of their axes. If it is not done the
         // swarm elements become elongated along X or Y.

         vec2 r = pixel - vec2 ( sin(theta), y );
         r = r + vec2 ( DELTA_X, DELTA_Y );
         r = r * swarm_scale;

         // Add the contribution this element makes to the colour
         // of the current pixel.  The  effect is a glow centered
         // on the element with its shape and the  way  it  glows
         // controlled by SWARM_ELEMENT.

         colour += SWARM_ELEMENT(i,r);

       }

    }

   // Output the colour for the pixel.  By  using the "length" of
   // the colour as the value for the alpha component pixels  are
   // increasingly transparent the further away they are from any
   // swarm element - allowing any background to show through.

   gl_FragColor = vec4 ( colour, length(colour) );

 }